package org.fastnate.generator.context;

import java.io.IOException;

import javax.persistence.SequenceGenerator;

import org.fastnate.generator.dialect.GeneratorDialect;
import org.fastnate.generator.statements.ColumnExpression;
import org.fastnate.generator.statements.CurrentSequenceValueExpression;
import org.fastnate.generator.statements.NextSequenceValueExpression;
import org.fastnate.generator.statements.PlainColumnExpression;
import org.fastnate.generator.statements.StatementsWriter;
import org.fastnate.generator.statements.TableStatement;

import lombok.AccessLevel;
import lombok.Getter;

/**
 * Stores the current value for a {@link SequenceGenerator}.
 *
 * @author Tobias Liefke
 */
@Getter
public class SequenceIdGenerator extends IdGenerator {

	/** The current database dialect. */
	private final GeneratorDialect dialect;

	/** The (optional) catalog that contains the schema of the sequence. */
	private final String catalog;

	/** The (optional) schema that contains the sequence. */
	private final String schema;

	/** The name of the sequence. */
	private final String sequenceName;

	/** The fully qualfied name of the sequence, including the optional catalog and schema name. */
	private final String qualifiedName;

	/** Indicates that the sequence is used when referencing existing IDs, instead of absolute IDs. */
	private final boolean relativeIds;

	/** The amount to increment by when allocating sequence numbers from the sequence. */
	private final int allocationSize;

	/** The value from which the sequence object is to start generating. */
	private long initialValue;

	@Getter(AccessLevel.NONE)
	private long nextValue;

	@Getter(AccessLevel.NONE)
	private long currentSequenceValue;

	/**
	 * Creates a new instance of {@link SequenceIdGenerator}.
	 *
	 * @param generator
	 *            the annotation that contains our settings
	 * @param dialect
	 *            the current database dialect
	 * @param relativeIds
	 *            indicates that the sequence is always used, instead of absolute IDs
	 */
	public SequenceIdGenerator(final SequenceGenerator generator, final GeneratorDialect dialect,
			final boolean relativeIds) {
		this.dialect = dialect;
		this.catalog = generator.catalog().length() == 0 ? null : generator.catalog();
		this.schema = generator.schema().length() == 0 ? null : generator.schema();
		this.sequenceName = generator.sequenceName();
		if (this.catalog == null) {
			if (this.schema == null) {
				this.qualifiedName = this.sequenceName;
			} else {
				this.qualifiedName = this.schema + '.' + this.sequenceName;
			}
		} else {
			ModelException.test(this.schema != null,
					"Catalog name '{}' found for sequence '{}' but schema name is missing.", this.catalog,
					this.sequenceName);
			this.qualifiedName = this.catalog + '.' + this.schema + '.' + this.sequenceName;
		}
		this.relativeIds = relativeIds;
		this.allocationSize = generator.allocationSize();
		this.nextValue = this.initialValue = generator.initialValue();
		this.currentSequenceValue = this.initialValue - 1;
	}

	@Override
	public void addNextValue(final TableStatement statement, final GeneratorColumn column, final Number value) {
		final ColumnExpression expression;
		if (this.dialect.isNextSequenceValueInInsertSupported() && this.currentSequenceValue <= value.longValue()) {
			if (this.currentSequenceValue < this.initialValue) {
				this.currentSequenceValue = this.initialValue;
			} else {
				this.currentSequenceValue += this.allocationSize;
			}
			expression = new NextSequenceValueExpression(this, this.currentSequenceValue - value.longValue());
		} else {
			expression = new CurrentSequenceValueExpression(this, this.currentSequenceValue - value.longValue());
		}
		statement.setColumnValue(column, expression);
	}

	@Override
	public void alignNextValue(final StatementsWriter writer) throws IOException {
		if (!this.relativeIds && this.nextValue > this.initialValue) {
			if (this.currentSequenceValue >= this.nextValue || this.currentSequenceValue < this.initialValue) {
				final long currentValue = this.currentSequenceValue;
				this.currentSequenceValue = this.nextValue - 1;
				this.dialect.adjustNextSequenceValue(writer, this.qualifiedName, currentValue,
						this.currentSequenceValue + this.allocationSize, this.allocationSize);
			}
		}
	}

	@Override
	public long createNextValue() {
		return this.nextValue++;
	}

	@Override
	public void createPreInsertStatements(final StatementsWriter writer) throws IOException {
		if (!this.dialect.isNextSequenceValueInInsertSupported() && this.currentSequenceValue <= this.nextValue) {
			if (this.currentSequenceValue < this.initialValue) {
				this.currentSequenceValue = this.initialValue;
			} else {
				this.currentSequenceValue += this.allocationSize;
			}
			writer.writePlainStatement(this.dialect,
					this.dialect.buildNextSequenceValue(this.qualifiedName, this.allocationSize));
		}
	}

	@Override
	public long getCurrentValue() {
		return this.nextValue - 1;
	}

	@Override
	public ColumnExpression getExpression(final GeneratorTable entityTable, final GeneratorColumn column,
			final Number targetId, final boolean whereExpression) {
		if (!whereExpression || this.dialect.isSequenceInWhereSupported()) {
			return new CurrentSequenceValueExpression(this, this.currentSequenceValue - targetId.longValue());
		}

		final long diff = this.nextValue - 1 - targetId.longValue();
		return new PlainColumnExpression("(SELECT max(" + column.getName() + ')' + (diff == 0 ? "" : " - " + diff)
				+ " FROM " + entityTable.getQualifiedName() + ')');
	}

	@Override
	public boolean isPostIncrement() {
		return false;
	}

	@Override
	public void setCurrentValue(final long currentValue) {
		this.nextValue = this.initialValue = currentValue + 1;
		this.currentSequenceValue = currentValue;
	}

}
